/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.HAUtil;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.LayoutVersion;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.common.InconsistentFSStateException;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.common.Util;
import org.apache.hadoop.hdfs.server.namenode.CheckpointFaultInjector;
import org.apache.hadoop.hdfs.server.namenode.CheckpointSignature;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLog;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.FSImageCompression;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormat;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormatProtobuf;
import org.apache.hadoop.hdfs.server.namenode.FSImagePreTransactionalStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FSImageStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FSImageTransactionalStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.IllegalReservedPathException;
import org.apache.hadoop.hdfs.server.namenode.MetaRecoveryContext;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NNStorageRetentionManager;
import org.apache.hadoop.hdfs.server.namenode.NNUpgradeUtil;
import org.apache.hadoop.hdfs.server.namenode.NameNode;
import org.apache.hadoop.hdfs.server.namenode.NameNodeLayoutVersion;
import org.apache.hadoop.hdfs.server.namenode.SaveNamespaceCancelledException;
import org.apache.hadoop.hdfs.server.namenode.SaveNamespaceContext;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.Phase;
import org.apache.hadoop.hdfs.server.namenode.startupprogress.StartupProgress;
import org.apache.hadoop.hdfs.server.protocol.CheckpointCommand;
import org.apache.hadoop.hdfs.server.protocol.NamenodeCommand;
import org.apache.hadoop.hdfs.server.protocol.NamenodeRegistration;
import org.apache.hadoop.hdfs.server.protocol.NamespaceInfo;
import org.apache.hadoop.hdfs.util.Canceler;
import org.apache.hadoop.hdfs.util.MD5FileUtils;
import org.apache.hadoop.io.MD5Hash;
import org.apache.hadoop.util.Time;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class FSImage
implements Closeable {
    public static final Log LOG = LogFactory.getLog((String)FSImage.class.getName());
    protected FSEditLog editLog = null;
    private boolean isUpgradeFinalized = false;
    protected NNStorage storage;
    protected long lastAppliedTxId = 0L;
    private final Configuration conf;
    protected NNStorageRetentionManager archivalManager;
    private int quotaInitThreads;
    private Set<Storage.StorageDirectory> newDirs = null;
    private final Set<Long> currentlyCheckpointing = Collections.synchronizedSet(new HashSet());

    public FSImage(Configuration conf) throws IOException {
        this(conf, FSNamesystem.getNamespaceDirs(conf), FSNamesystem.getNamespaceEditsDirs(conf));
    }

    protected FSImage(Configuration conf, Collection<URI> imageDirs, List<URI> editsDirs) throws IOException {
        this.conf = conf;
        this.storage = new NNStorage(conf, imageDirs, editsDirs);
        if (conf.getBoolean("dfs.namenode.name.dir.restore", false)) {
            this.storage.setRestoreFailedStorage(true);
        }
        this.editLog = FSEditLog.newInstance(conf, this.storage, editsDirs);
        this.archivalManager = new NNStorageRetentionManager(conf, this.storage, this.editLog);
    }

    void format(FSNamesystem fsn, String clusterId) throws IOException {
        long fileCount = fsn.getTotalFiles();
        Preconditions.checkState((fileCount == 1L ? 1 : 0) != 0, (Object)("FSImage.format should be called with an uninitialized namesystem, has " + fileCount + " files"));
        NamespaceInfo ns = NNStorage.newNamespaceInfo();
        LOG.info((Object)("Allocated new BlockPoolId: " + ns.getBlockPoolID()));
        ns.clusterID = clusterId;
        this.storage.format(ns);
        this.editLog.formatNonFileJournals(ns);
        this.saveFSImageInAllDirs(fsn, 0L);
    }

    boolean confirmFormat(boolean force, boolean interactive) throws IOException {
        ArrayList confirms = Lists.newArrayList();
        for (Storage.StorageDirectory sd : this.storage.dirIterable(null)) {
            confirms.add(sd);
        }
        confirms.addAll(this.editLog.getFormatConfirmables());
        return Storage.confirmFormat(confirms, force, interactive);
    }

    boolean recoverTransitionRead(HdfsServerConstants.StartupOption startOpt, FSNamesystem target, MetaRecoveryContext recovery) throws IOException {
        assert (startOpt != HdfsServerConstants.StartupOption.FORMAT) : "NameNode formatting should be performed before reading the image";
        Collection<URI> imageDirs = this.storage.getImageDirectories();
        Collection<URI> editsDirs = this.editLog.getEditURIs();
        if ((imageDirs.size() == 0 || editsDirs.size() == 0) && startOpt != HdfsServerConstants.StartupOption.IMPORT) {
            throw new IOException("All specified directories are not accessible or do not exist.");
        }
        HashMap<Storage.StorageDirectory, Storage.StorageState> dataDirStates = new HashMap<Storage.StorageDirectory, Storage.StorageState>();
        boolean isFormatted = FSImage.recoverStorageDirs(startOpt, this.storage, dataDirStates);
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Data dir states:\n  " + Joiner.on((String)"\n  ").withKeyValueSeparator(": ").join(dataDirStates)));
        }
        if (!isFormatted && startOpt != HdfsServerConstants.StartupOption.ROLLBACK && startOpt != HdfsServerConstants.StartupOption.IMPORT) {
            throw new IOException("NameNode is not formatted.");
        }
        int layoutVersion = this.storage.getLayoutVersion();
        if (startOpt == HdfsServerConstants.StartupOption.METADATAVERSION) {
            System.out.println("HDFS Image Version: " + layoutVersion);
            System.out.println("Software format version: " + HdfsConstants.NAMENODE_LAYOUT_VERSION);
            return false;
        }
        if (layoutVersion < -3) {
            NNStorage.checkVersionUpgradable(this.storage.getLayoutVersion());
        }
        if (startOpt != HdfsServerConstants.StartupOption.UPGRADE && startOpt != HdfsServerConstants.StartupOption.UPGRADEONLY && !HdfsServerConstants.RollingUpgradeStartupOption.STARTED.matches(startOpt) && layoutVersion < -3 && layoutVersion != HdfsConstants.NAMENODE_LAYOUT_VERSION) {
            throw new IOException("\nFile system image contains an old layout version " + this.storage.getLayoutVersion() + ".\nAn upgrade to version " + HdfsConstants.NAMENODE_LAYOUT_VERSION + " is required.\n" + "Please restart NameNode with the \"" + HdfsServerConstants.RollingUpgradeStartupOption.STARTED.getOptionString() + "\" option if a rolling upgrade is already started;" + " or restart NameNode with the \"" + HdfsServerConstants.StartupOption.UPGRADE.getName() + "\" option to start" + " a new upgrade.");
        }
        this.storage.processStartupOptionsForUpgrade(startOpt, layoutVersion);
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageDirectory sd = it.next();
            Storage.StorageState curState = (Storage.StorageState)((Object)dataDirStates.get(sd));
            switch (curState) {
                case NON_EXISTENT: {
                    throw new IOException((Object)((Object)Storage.StorageState.NON_EXISTENT) + " state cannot be here");
                }
                case NOT_FORMATTED: {
                    LOG.info((Object)("Storage directory " + sd.getRoot() + " is not formatted."));
                    LOG.info((Object)"Formatting ...");
                    sd.clearDirectory();
                    if (!target.isHaEnabled()) break;
                    if (this.newDirs == null) {
                        this.newDirs = new HashSet<Storage.StorageDirectory>();
                    }
                    this.newDirs.add(sd);
                    break;
                }
            }
        }
        switch (startOpt) {
            case UPGRADE: 
            case UPGRADEONLY: {
                this.doUpgrade(target);
                return false;
            }
            case IMPORT: {
                this.doImportCheckpoint(target);
                return false;
            }
            case ROLLBACK: {
                throw new AssertionError((Object)"Rollback is now a standalone command, NameNode should not be starting with this option.");
            }
        }
        return this.loadFSImage(target, startOpt, recovery);
    }

    private void initNewDirs() {
        if (this.newDirs == null) {
            return;
        }
        for (Storage.StorageDirectory sd : this.newDirs) {
            try {
                this.storage.writeProperties(sd);
                LOG.info((Object)("Wrote VERSION in the new storage, " + sd.getCurrentDir()));
            }
            catch (IOException e) {
                this.storage.reportErrorOnFile(sd.getVersionFile());
            }
        }
        this.newDirs.clear();
        this.newDirs = null;
    }

    public static boolean recoverStorageDirs(HdfsServerConstants.StartupOption startOpt, NNStorage storage, Map<Storage.StorageDirectory, Storage.StorageState> dataDirStates) throws IOException {
        boolean isFormatted = false;
        Iterator<Storage.StorageDirectory> it = storage.dirIterator();
        while (it.hasNext()) {
            Storage.StorageState curState;
            Storage.StorageDirectory sd = it.next();
            if (startOpt == HdfsServerConstants.StartupOption.METADATAVERSION) {
                storage.readProperties(sd);
                return true;
            }
            try {
                curState = sd.analyzeStorage(startOpt, storage);
                switch (curState) {
                    case NON_EXISTENT: {
                        throw new InconsistentFSStateException(sd.getRoot(), "storage directory does not exist or is not accessible.");
                    }
                    case NOT_FORMATTED: {
                        break;
                    }
                    case NORMAL: {
                        break;
                    }
                    default: {
                        sd.doRecover(curState);
                    }
                }
                if (curState != Storage.StorageState.NOT_FORMATTED && startOpt != HdfsServerConstants.StartupOption.ROLLBACK) {
                    storage.readProperties(sd, startOpt);
                    isFormatted = true;
                }
                if (startOpt == HdfsServerConstants.StartupOption.IMPORT && isFormatted) {
                    throw new IOException("Cannot import image from a checkpoint.  NameNode already contains an image in " + sd.getRoot());
                }
            }
            catch (IOException ioe) {
                sd.unlock();
                throw ioe;
            }
            dataDirStates.put(sd, curState);
        }
        return isFormatted;
    }

    public static void checkUpgrade(NNStorage storage) throws IOException {
        Iterator<Storage.StorageDirectory> it = storage.dirIterator(false);
        while (it.hasNext()) {
            Storage.StorageDirectory sd = it.next();
            if (!sd.getPreviousDir().exists()) continue;
            throw new InconsistentFSStateException(sd.getRoot(), "previous fs state should not exist during upgrade. Finalize or rollback first.");
        }
    }

    void checkUpgrade() throws IOException {
        FSImage.checkUpgrade(this.storage);
    }

    public boolean hasRollbackFSImage() throws IOException {
        FSImageTransactionalStorageInspector inspector = new FSImageTransactionalStorageInspector(EnumSet.of(NNStorage.NameNodeFile.IMAGE_ROLLBACK));
        this.storage.inspectStorageDirs(inspector);
        try {
            List<FSImageStorageInspector.FSImageFile> images = ((FSImageStorageInspector)inspector).getLatestImages();
            return images != null && !images.isEmpty();
        }
        catch (FileNotFoundException e) {
            return false;
        }
    }

    void doUpgrade(FSNamesystem target) throws IOException {
        Storage.StorageDirectory sd;
        this.checkUpgrade();
        this.loadFSImage(target, HdfsServerConstants.StartupOption.UPGRADE, null);
        target.checkRollingUpgrade("upgrade namenode");
        long oldCTime = this.storage.getCTime();
        this.storage.cTime = Time.now();
        int oldLV = this.storage.getLayoutVersion();
        this.storage.layoutVersion = HdfsConstants.NAMENODE_LAYOUT_VERSION;
        List<Storage.StorageDirectory> errorSDs = Collections.synchronizedList(new ArrayList());
        assert (!this.editLog.isSegmentOpen()) : "Edits log must not be open.";
        LOG.info((Object)("Starting upgrade of local storage directories.\n   old LV = " + oldLV + "; old CTime = " + oldCTime + ".\n   new LV = " + this.storage.getLayoutVersion() + "; new CTime = " + this.storage.getCTime()));
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator(false);
        while (it.hasNext()) {
            sd = it.next();
            try {
                NNUpgradeUtil.doPreUpgrade(this.conf, sd);
            }
            catch (Exception e) {
                LOG.error((Object)("Failed to move aside pre-upgrade storage in image directory " + sd.getRoot()), (Throwable)e);
                errorSDs.add(sd);
            }
        }
        if (target.isHaEnabled()) {
            this.editLog.doPreUpgradeOfSharedLog();
        }
        this.storage.reportErrorsOnDirectories(errorSDs);
        errorSDs.clear();
        this.saveFSImageInAllDirs(target, this.editLog.getLastWrittenTxId());
        if (target.isHaEnabled()) {
            this.editLog.doUpgradeOfSharedLog();
        }
        it = this.storage.dirIterator(false);
        while (it.hasNext()) {
            sd = it.next();
            try {
                NNUpgradeUtil.doUpgrade(sd, this.storage);
            }
            catch (IOException ioe) {
                errorSDs.add(sd);
            }
        }
        this.storage.reportErrorsOnDirectories(errorSDs);
        this.isUpgradeFinalized = false;
        if (!this.storage.getRemovedStorageDirs().isEmpty()) {
            throw new IOException("Upgrade failed in " + this.storage.getRemovedStorageDirs().size() + " storage directory(ies), previously logged.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void doRollback(FSNamesystem fsns) throws IOException {
        boolean canRollback = false;
        try (FSImage prevState = new FSImage(this.conf);){
            Storage.StorageDirectory sd;
            prevState.getStorage().layoutVersion = HdfsConstants.NAMENODE_LAYOUT_VERSION;
            Iterator<Storage.StorageDirectory> it = this.storage.dirIterator(false);
            while (it.hasNext()) {
                sd = it.next();
                if (!NNUpgradeUtil.canRollBack(sd, this.storage, prevState.getStorage(), HdfsConstants.NAMENODE_LAYOUT_VERSION)) continue;
                LOG.info((Object)("Can perform rollback for " + sd));
                canRollback = true;
            }
            if (fsns.isHaEnabled()) {
                this.editLog.initJournalsForWrite();
                boolean canRollBackSharedEditLog = this.editLog.canRollBackSharedLog(prevState.getStorage(), HdfsConstants.NAMENODE_LAYOUT_VERSION);
                if (canRollBackSharedEditLog) {
                    LOG.info((Object)"Can perform rollback for shared edit log.");
                    canRollback = true;
                }
            }
            if (!canRollback) {
                throw new IOException("Cannot rollback. None of the storage directories contain previous fs state.");
            }
            it = this.storage.dirIterator(false);
            while (it.hasNext()) {
                sd = it.next();
                LOG.info((Object)("Rolling back storage directory " + sd.getRoot() + ".\n   new LV = " + prevState.getStorage().getLayoutVersion() + "; new CTime = " + prevState.getStorage().getCTime()));
                NNUpgradeUtil.doRollBack(sd);
            }
            if (fsns.isHaEnabled()) {
                this.editLog.doRollback();
            }
            this.isUpgradeFinalized = true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void doImportCheckpoint(FSNamesystem target) throws IOException {
        Collection<URI> checkpointDirs = FSImage.getCheckpointDirs(this.conf, null);
        List<URI> checkpointEditsDirs = FSImage.getCheckpointEditsDirs(this.conf, null);
        if (checkpointDirs == null || checkpointDirs.isEmpty()) {
            throw new IOException("Cannot import image from a checkpoint. \"dfs.namenode.checkpoint.dir\" is not set.");
        }
        if (checkpointEditsDirs == null || checkpointEditsDirs.isEmpty()) {
            throw new IOException("Cannot import image from a checkpoint. \"dfs.namenode.checkpoint.dir\" is not set.");
        }
        FSImage realImage = target.getFSImage();
        try (FSImage ckptImage = new FSImage(this.conf, checkpointDirs, checkpointEditsDirs);){
            ckptImage.recoverTransitionRead(HdfsServerConstants.StartupOption.REGULAR, target, null);
        }
        realImage.getStorage().setStorageInfo(ckptImage.getStorage());
        realImage.getEditLog().setNextTxId(ckptImage.getEditLog().getLastWrittenTxId() + 1L);
        realImage.initEditLog(HdfsServerConstants.StartupOption.IMPORT);
        realImage.getStorage().setBlockPoolID(ckptImage.getBlockPoolID());
        this.saveNamespace(target);
        this.getStorage().writeAll();
    }

    void finalizeUpgrade(boolean finalizeEditLog) throws IOException {
        LOG.info((Object)("Finalizing upgrade for local dirs. " + (this.storage.getLayoutVersion() == 0 ? "" : "\n   cur LV = " + this.storage.getLayoutVersion() + "; cur CTime = " + this.storage.getCTime())));
        Iterator<Storage.StorageDirectory> it = this.storage.dirIterator(false);
        while (it.hasNext()) {
            Storage.StorageDirectory sd = it.next();
            NNUpgradeUtil.doFinalize(sd);
        }
        if (finalizeEditLog) {
            this.editLog.doFinalizeOfSharedLog();
        }
        this.isUpgradeFinalized = true;
    }

    boolean isUpgradeFinalized() {
        return this.isUpgradeFinalized;
    }

    public FSEditLog getEditLog() {
        return this.editLog;
    }

    @VisibleForTesting
    public void setEditLogForTesting(FSEditLog newLog) {
        newLog.restart();
        this.editLog = newLog;
    }

    void openEditLogForWrite() throws IOException {
        assert (this.editLog != null) : "editLog must be initialized";
        this.editLog.openForWrite();
        this.storage.writeTransactionIdFileToStorage(this.editLog.getCurSegmentTxId());
    }

    void reloadFromImageFile(File file, FSNamesystem target) throws IOException {
        target.clear();
        LOG.debug((Object)("Reloading namespace from " + file));
        this.loadFSImage(file, target, null, false);
    }

    private boolean loadFSImage(FSNamesystem target, HdfsServerConstants.StartupOption startOpt, MetaRecoveryContext recovery) throws IOException {
        boolean rollingRollback = HdfsServerConstants.RollingUpgradeStartupOption.ROLLBACK.matches(startOpt);
        EnumSet<NNStorage.NameNodeFile> nnfs = rollingRollback ? EnumSet.of(NNStorage.NameNodeFile.IMAGE_ROLLBACK) : EnumSet.of(NNStorage.NameNodeFile.IMAGE, NNStorage.NameNodeFile.IMAGE_ROLLBACK);
        FSImageStorageInspector inspector = this.storage.readAndInspectDirs(nnfs, startOpt);
        this.isUpgradeFinalized = inspector.isUpgradeFinalized();
        List<FSImageStorageInspector.FSImageFile> imageFiles = inspector.getLatestImages();
        StartupProgress prog = NameNode.getStartupProgress();
        prog.beginPhase(Phase.LOADING_FSIMAGE);
        File phaseFile = imageFiles.get(0).getFile();
        prog.setFile(Phase.LOADING_FSIMAGE, phaseFile.getAbsolutePath());
        prog.setSize(Phase.LOADING_FSIMAGE, phaseFile.length());
        boolean needToSave = inspector.needToSave();
        Iterable<EditLogInputStream> editStreams = null;
        this.initEditLog(startOpt);
        if (NameNodeLayoutVersion.supports(LayoutVersion.Feature.TXID_BASED_LAYOUT, this.getLayoutVersion())) {
            long toAtLeastTxId;
            long l = toAtLeastTxId = this.editLog.isOpenForWrite() ? inspector.getMaxSeenTxId() : 0L;
            if (rollingRollback) {
                toAtLeastTxId = imageFiles.get(0).getCheckpointTxId() + 2L;
            }
            editStreams = this.editLog.selectInputStreams(imageFiles.get(0).getCheckpointTxId() + 1L, toAtLeastTxId, recovery, false);
        } else {
            editStreams = FSImagePreTransactionalStorageInspector.getEditLogStreams(this.storage);
        }
        int maxOpSize = this.conf.getInt("dfs.namenode.max.op.size", 0x3200000);
        for (EditLogInputStream elis : editStreams) {
            elis.setMaxOpSize(maxOpSize);
        }
        for (EditLogInputStream l : editStreams) {
            LOG.debug((Object)("Planning to load edit log stream: " + l));
        }
        if (!editStreams.iterator().hasNext()) {
            LOG.info((Object)"No edit log streams selected.");
        }
        Exception le = null;
        FSImageStorageInspector.FSImageFile imageFile = null;
        for (int i = 0; i < imageFiles.size(); ++i) {
            try {
                imageFile = imageFiles.get(i);
                this.loadFSImageFile(target, recovery, imageFile, startOpt);
                break;
            }
            catch (IllegalReservedPathException ie) {
                throw new IOException("Failed to load image from " + imageFile, ie);
            }
            catch (Exception e) {
                le = e;
                LOG.error((Object)("Failed to load image from " + imageFile), (Throwable)e);
                target.clear();
                imageFile = null;
                continue;
            }
        }
        if (imageFile == null) {
            FSEditLog.closeAllStreams(editStreams);
            throw new IOException("Failed to load FSImage file, see error(s) above for more info.");
        }
        prog.endPhase(Phase.LOADING_FSIMAGE);
        if (!rollingRollback) {
            long txnsAdvanced = this.loadEdits(editStreams, target, startOpt, recovery);
            needToSave |= this.needsResaveBasedOnStaleCheckpoint(imageFile.getFile(), txnsAdvanced);
            if (HdfsServerConstants.RollingUpgradeStartupOption.DOWNGRADE.matches(startOpt)) {
                this.renameCheckpoint(NNStorage.NameNodeFile.IMAGE_ROLLBACK, NNStorage.NameNodeFile.IMAGE);
            }
        } else {
            this.rollingRollback(this.lastAppliedTxId + 1L, imageFiles.get(0).getCheckpointTxId());
            needToSave = false;
        }
        this.editLog.setNextTxId(this.lastAppliedTxId + 1L);
        return needToSave;
    }

    private void rollingRollback(long discardSegmentTxId, long ckptId) throws IOException {
        this.editLog.discardSegments(discardSegmentTxId);
        this.renameCheckpoint(ckptId, NNStorage.NameNodeFile.IMAGE_ROLLBACK, NNStorage.NameNodeFile.IMAGE, true);
        this.archivalManager.purgeCheckpoinsAfter(NNStorage.NameNodeFile.IMAGE, ckptId);
        this.archivalManager.purgeCheckpoints(NNStorage.NameNodeFile.IMAGE_ROLLBACK);
        String nameserviceId = DFSUtil.getNamenodeNameServiceId(this.conf);
        if (HAUtil.isHAEnabled(this.conf, nameserviceId)) {
            this.editLog.close();
            this.editLog.initSharedJournalsForRead();
        }
    }

    void loadFSImageFile(FSNamesystem target, MetaRecoveryContext recovery, FSImageStorageInspector.FSImageFile imageFile, HdfsServerConstants.StartupOption startupOption) throws IOException {
        LOG.info((Object)("Planning to load image: " + imageFile));
        Storage.StorageDirectory sdForProperties = imageFile.sd;
        this.storage.readProperties(sdForProperties, startupOption);
        if (NameNodeLayoutVersion.supports(LayoutVersion.Feature.TXID_BASED_LAYOUT, this.getLayoutVersion())) {
            boolean isRollingRollback = HdfsServerConstants.RollingUpgradeStartupOption.ROLLBACK.matches(startupOption);
            this.loadFSImage(imageFile.getFile(), target, recovery, isRollingRollback);
        } else if (NameNodeLayoutVersion.supports(LayoutVersion.Feature.FSIMAGE_CHECKSUM, this.getLayoutVersion())) {
            String md5 = this.storage.getDeprecatedProperty("imageMD5Digest");
            if (md5 == null) {
                throw new InconsistentFSStateException(sdForProperties.getRoot(), "Message digest property imageMD5Digest not set for storage directory " + sdForProperties.getRoot());
            }
            this.loadFSImage(imageFile.getFile(), new MD5Hash(md5), target, recovery, false);
        } else {
            this.loadFSImage(imageFile.getFile(), null, target, recovery, false);
        }
    }

    public void initEditLog(HdfsServerConstants.StartupOption startOpt) throws IOException {
        Preconditions.checkState((this.getNamespaceID() != 0 ? 1 : 0) != 0, (Object)"Must know namespace ID before initting edit log");
        String nameserviceId = DFSUtil.getNamenodeNameServiceId(this.conf);
        if (!HAUtil.isHAEnabled(this.conf, nameserviceId)) {
            this.editLog.initJournalsForWrite();
            this.editLog.recoverUnclosedStreams();
        } else if (HAUtil.isHAEnabled(this.conf, nameserviceId) && (startOpt == HdfsServerConstants.StartupOption.UPGRADE || startOpt == HdfsServerConstants.StartupOption.UPGRADEONLY || HdfsServerConstants.RollingUpgradeStartupOption.ROLLBACK.matches(startOpt))) {
            this.editLog.initJournalsForWrite();
            if (startOpt == HdfsServerConstants.StartupOption.UPGRADE || startOpt == HdfsServerConstants.StartupOption.UPGRADEONLY) {
                long sharedLogCTime = this.editLog.getSharedLogCTime();
                if (this.storage.getCTime() < sharedLogCTime) {
                    throw new IOException("It looks like the shared log is already being upgraded but this NN has not been upgraded yet. You should restart this NameNode with the '" + HdfsServerConstants.StartupOption.BOOTSTRAPSTANDBY.getName() + "' option to bring " + "this NN in sync with the other.");
                }
            }
            this.editLog.recoverUnclosedStreams();
        } else {
            this.editLog.initSharedJournalsForRead();
        }
    }

    private boolean needsResaveBasedOnStaleCheckpoint(File imageFile, long numEditsLoaded) {
        long checkpointPeriod = this.conf.getLong("dfs.namenode.checkpoint.period", 3600L);
        long checkpointTxnCount = this.conf.getLong("dfs.namenode.checkpoint.txns", 1000000L);
        long checkpointAge = Time.now() - imageFile.lastModified();
        return checkpointAge > checkpointPeriod * 1000L || numEditsLoaded > checkpointTxnCount;
    }

    public long loadEdits(Iterable<EditLogInputStream> editStreams, FSNamesystem target) throws IOException {
        return this.loadEdits(editStreams, target, null, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long loadEdits(Iterable<EditLogInputStream> editStreams, FSNamesystem target, HdfsServerConstants.StartupOption startOpt, MetaRecoveryContext recovery) throws IOException {
        LOG.debug((Object)("About to load edits:\n  " + Joiner.on((String)"\n  ").join(editStreams)));
        StartupProgress prog = NameNode.getStartupProgress();
        prog.beginPhase(Phase.LOADING_EDITS);
        long prevLastAppliedTxId = this.lastAppliedTxId;
        try {
            FSEditLogLoader loader = new FSEditLogLoader(target, this.lastAppliedTxId);
            for (EditLogInputStream editIn : editStreams) {
                LOG.info((Object)("Reading " + editIn + " expecting start txid #" + (this.lastAppliedTxId + 1L)));
                try {
                    loader.loadFSEdits(editIn, this.lastAppliedTxId + 1L, startOpt, recovery);
                }
                finally {
                    this.lastAppliedTxId = loader.getLastAppliedTxId();
                }
                if (editIn.getLastTxId() == -12345L) continue;
                this.lastAppliedTxId = editIn.getLastTxId();
            }
        }
        finally {
            FSEditLog.closeAllStreams(editStreams);
        }
        prog.endPhase(Phase.LOADING_EDITS);
        return this.lastAppliedTxId - prevLastAppliedTxId;
    }

    private void loadFSImage(File imageFile, FSNamesystem target, MetaRecoveryContext recovery, boolean requireSameLayoutVersion) throws IOException {
        MD5Hash expectedMD5 = MD5FileUtils.readStoredMd5ForFile(imageFile);
        if (expectedMD5 == null) {
            throw new IOException("No MD5 file found corresponding to image file " + imageFile);
        }
        this.loadFSImage(imageFile, expectedMD5, target, recovery, requireSameLayoutVersion);
    }

    private void loadFSImage(File curFile, MD5Hash expectedMd5, FSNamesystem target, MetaRecoveryContext recovery, boolean requireSameLayoutVersion) throws IOException {
        target.setBlockPoolId(this.getBlockPoolID());
        FSImageFormat.LoaderDelegator loader = FSImageFormat.newLoader(this.conf, target);
        loader.load(curFile, requireSameLayoutVersion);
        MD5Hash readImageMd5 = loader.getLoadedImageMd5();
        if (expectedMd5 != null && !expectedMd5.equals((Object)readImageMd5)) {
            throw new IOException("Image file " + curFile + " is corrupt with MD5 checksum of " + readImageMd5 + " but expecting " + expectedMd5);
        }
        long txId = loader.getLoadedImageTxId();
        LOG.info((Object)("Loaded image for txid " + txId + " from " + curFile));
        this.lastAppliedTxId = txId;
        this.storage.setMostRecentCheckpointInfo(txId, curFile.lastModified());
    }

    void saveFSImage(SaveNamespaceContext context, Storage.StorageDirectory sd, NNStorage.NameNodeFile dstType) throws IOException {
        long txid = context.getTxId();
        File newFile = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE_NEW, txid);
        File dstFile = NNStorage.getStorageFile(sd, dstType, txid);
        FSImageFormatProtobuf.Saver saver = new FSImageFormatProtobuf.Saver(context);
        FSImageCompression compression = FSImageCompression.createCompression(this.conf);
        saver.save(newFile, compression);
        MD5FileUtils.saveMD5File(dstFile, saver.getSavedDigest());
        this.storage.setMostRecentCheckpointInfo(txid, Time.now());
    }

    public void saveLegacyOIVImage(FSNamesystem source, String targetDir, Canceler canceler) throws IOException {
        FSImageCompression compression = FSImageCompression.createCompression(this.conf);
        long txid = this.getCorrectLastAppliedOrWrittenTxId();
        SaveNamespaceContext ctx = new SaveNamespaceContext(source, txid, canceler);
        FSImageFormat.Saver saver = new FSImageFormat.Saver(ctx);
        String imageFileName = NNStorage.getLegacyOIVImageFileName(txid);
        File imageFile = new File(targetDir, imageFileName);
        saver.save(imageFile, compression);
        this.archivalManager.purgeOldLegacyOIVImages(targetDir, txid);
    }

    private void waitForThreads(List<Thread> threads) {
        for (Thread thread : threads) {
            while (thread.isAlive()) {
                try {
                    thread.join();
                }
                catch (InterruptedException iex) {
                    LOG.error((Object)("Caught interrupted exception while waiting for thread " + thread.getName() + " to finish. Retrying join"));
                }
            }
        }
    }

    public synchronized void updateStorageVersion() throws IOException {
        this.storage.writeAll();
    }

    public synchronized void saveNamespace(FSNamesystem source) throws IOException {
        this.saveNamespace(source, NNStorage.NameNodeFile.IMAGE, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void saveNamespace(FSNamesystem source, NNStorage.NameNodeFile nnf, Canceler canceler) throws IOException {
        long imageTxId;
        assert (this.editLog != null) : "editLog must be initialized";
        LOG.info((Object)"Save namespace ...");
        this.storage.attemptRestoreRemovedStorage();
        boolean editLogWasOpen = this.editLog.isSegmentOpen();
        if (editLogWasOpen) {
            this.editLog.endCurrentLogSegment(true);
        }
        if (!this.addToCheckpointing(imageTxId = this.getCorrectLastAppliedOrWrittenTxId())) {
            throw new IOException("FS image is being downloaded from another NN at txid " + imageTxId);
        }
        try {
            try {
                this.saveFSImageInAllDirs(source, nnf, imageTxId, canceler);
                this.storage.writeAll();
            }
            finally {
                if (editLogWasOpen) {
                    this.editLog.startLogSegment(imageTxId + 1L, true);
                    this.storage.writeTransactionIdFileToStorage(imageTxId + 1L);
                }
            }
        }
        finally {
            this.removeFromCheckpointing(imageTxId);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected synchronized void saveFSImageInAllDirs(FSNamesystem source, long txid) throws IOException {
        if (!this.addToCheckpointing(txid)) {
            throw new IOException("FS image is being downloaded from another NN");
        }
        try {
            this.saveFSImageInAllDirs(source, NNStorage.NameNodeFile.IMAGE, txid, null);
        }
        finally {
            this.removeFromCheckpointing(txid);
        }
    }

    public boolean addToCheckpointing(long txid) {
        return this.currentlyCheckpointing.add(txid);
    }

    public void removeFromCheckpointing(long txid) {
        this.currentlyCheckpointing.remove(txid);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void saveFSImageInAllDirs(FSNamesystem source, NNStorage.NameNodeFile nnf, long txid, Canceler canceler) throws IOException {
        StartupProgress prog = NameNode.getStartupProgress();
        prog.beginPhase(Phase.SAVING_CHECKPOINT);
        if (this.storage.getNumStorageDirs(NNStorage.NameNodeDirType.IMAGE) == 0) {
            throw new IOException("No image directories available!");
        }
        if (canceler == null) {
            canceler = new Canceler();
        }
        SaveNamespaceContext ctx = new SaveNamespaceContext(source, txid, canceler);
        try {
            ArrayList<Thread> saveThreads = new ArrayList<Thread>();
            Iterator<Storage.StorageDirectory> it = this.storage.dirIterator(NNStorage.NameNodeDirType.IMAGE);
            while (it.hasNext()) {
                Storage.StorageDirectory sd = it.next();
                FSImageSaver saver = new FSImageSaver(ctx, sd, nnf);
                Thread saveThread = new Thread((Runnable)saver, saver.toString());
                saveThreads.add(saveThread);
                saveThread.start();
            }
            this.waitForThreads(saveThreads);
            saveThreads.clear();
            this.storage.reportErrorsOnDirectories(ctx.getErrorSDs());
            if (this.storage.getNumStorageDirs(NNStorage.NameNodeDirType.IMAGE) == 0) {
                throw new IOException("Failed to save in any storage directories while saving namespace.");
            }
            if (canceler.isCancelled()) {
                this.deleteCancelledCheckpoint(txid);
                ctx.checkCancelled();
                assert (false) : "should have thrown above!";
            }
            this.renameCheckpoint(txid, NNStorage.NameNodeFile.IMAGE_NEW, nnf, false);
            this.purgeOldStorage(nnf);
            this.archivalManager.purgeCheckpoints(NNStorage.NameNodeFile.IMAGE_NEW);
        }
        finally {
            ctx.markComplete();
            ctx = null;
        }
        prog.endPhase(Phase.SAVING_CHECKPOINT);
    }

    void purgeOldStorage(NNStorage.NameNodeFile nnf) {
        try {
            this.archivalManager.purgeOldStorage(nnf);
        }
        catch (Exception e) {
            LOG.warn((Object)("Unable to purge old storage " + nnf.getName()), (Throwable)e);
        }
    }

    private void renameCheckpoint(long txid, NNStorage.NameNodeFile fromNnf, NNStorage.NameNodeFile toNnf, boolean renameMD5) throws IOException {
        ArrayList al = null;
        for (Storage.StorageDirectory sd : this.storage.dirIterable(NNStorage.NameNodeDirType.IMAGE)) {
            try {
                this.renameImageFileInDir(sd, fromNnf, toNnf, txid, renameMD5);
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Unable to rename checkpoint in " + sd), (Throwable)ioe);
                if (al == null) {
                    al = Lists.newArrayList();
                }
                al.add(sd);
            }
        }
        if (al != null) {
            this.storage.reportErrorsOnDirectories(al);
        }
    }

    void renameCheckpoint(NNStorage.NameNodeFile fromNnf, NNStorage.NameNodeFile toNnf) throws IOException {
        ArrayList al = null;
        FSImageTransactionalStorageInspector inspector = new FSImageTransactionalStorageInspector(EnumSet.of(fromNnf));
        this.storage.inspectStorageDirs(inspector);
        for (FSImageStorageInspector.FSImageFile image : inspector.getFoundImages()) {
            try {
                this.renameImageFileInDir(image.sd, fromNnf, toNnf, image.txId, true);
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Unable to rename checkpoint in " + image.sd), (Throwable)ioe);
                if (al == null) {
                    al = Lists.newArrayList();
                }
                al.add(image.sd);
            }
        }
        if (al != null) {
            this.storage.reportErrorsOnDirectories(al);
        }
    }

    private void deleteCancelledCheckpoint(long txid) throws IOException {
        ArrayList al = Lists.newArrayList();
        for (Storage.StorageDirectory sd : this.storage.dirIterable(NNStorage.NameNodeDirType.IMAGE)) {
            File ckpt = NNStorage.getStorageFile(sd, NNStorage.NameNodeFile.IMAGE_NEW, txid);
            if (!ckpt.exists() || ckpt.delete()) continue;
            LOG.warn((Object)("Unable to delete cancelled checkpoint in " + sd));
            al.add(sd);
        }
        this.storage.reportErrorsOnDirectories(al);
    }

    private void renameImageFileInDir(Storage.StorageDirectory sd, NNStorage.NameNodeFile fromNnf, NNStorage.NameNodeFile toNnf, long txid, boolean renameMD5) throws IOException {
        File fromFile = NNStorage.getStorageFile(sd, fromNnf, txid);
        File toFile = NNStorage.getStorageFile(sd, toNnf, txid);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("renaming  " + fromFile.getAbsolutePath() + " to " + toFile.getAbsolutePath()));
        }
        if (!(fromFile.renameTo(toFile) || toFile.delete() && fromFile.renameTo(toFile))) {
            throw new IOException("renaming  " + fromFile.getAbsolutePath() + " to " + toFile.getAbsolutePath() + " FAILED");
        }
        if (renameMD5) {
            MD5FileUtils.renameMD5File(fromFile, toFile);
        }
    }

    CheckpointSignature rollEditLog() throws IOException {
        this.getEditLog().rollEditLog();
        this.storage.writeTransactionIdFileToStorage(this.getEditLog().getCurSegmentTxId());
        return new CheckpointSignature(this);
    }

    NamenodeCommand startCheckpoint(NamenodeRegistration bnReg, NamenodeRegistration nnReg) throws IOException {
        LOG.info((Object)("Start checkpoint at txid " + this.getEditLog().getLastWrittenTxId()));
        String msg = null;
        if (bnReg.getNamespaceID() != this.storage.getNamespaceID()) {
            msg = "Name node " + bnReg.getAddress() + " has incompatible namespace id: " + bnReg.getNamespaceID() + " expected: " + this.storage.getNamespaceID();
        } else if (bnReg.isRole(HdfsServerConstants.NamenodeRole.NAMENODE)) {
            msg = "Name node " + bnReg.getAddress() + " role " + (Object)((Object)bnReg.getRole()) + ": checkpoint is not allowed.";
        } else if (bnReg.getLayoutVersion() < this.storage.getLayoutVersion() || bnReg.getLayoutVersion() == this.storage.getLayoutVersion() && bnReg.getCTime() > this.storage.getCTime()) {
            msg = "Name node " + bnReg.getAddress() + " has newer image layout version: LV = " + bnReg.getLayoutVersion() + " cTime = " + bnReg.getCTime() + ". Current version: LV = " + this.storage.getLayoutVersion() + " cTime = " + this.storage.getCTime();
        }
        if (msg != null) {
            LOG.error((Object)msg);
            return new NamenodeCommand(50);
        }
        boolean needToReturnImg = true;
        if (this.storage.getNumStorageDirs(NNStorage.NameNodeDirType.IMAGE) == 0) {
            needToReturnImg = false;
        }
        CheckpointSignature sig = this.rollEditLog();
        return new CheckpointCommand(sig, needToReturnImg);
    }

    void endCheckpoint(CheckpointSignature sig) throws IOException {
        LOG.info((Object)("End checkpoint at txid " + this.getEditLog().getLastWrittenTxId()));
        sig.validateStorageInfo(this);
    }

    public synchronized void saveDigestAndRenameCheckpointImage(NNStorage.NameNodeFile nnf, long txid, MD5Hash digest) throws IOException {
        ArrayList badSds = Lists.newArrayList();
        for (Storage.StorageDirectory sd : this.storage.dirIterable(NNStorage.NameNodeDirType.IMAGE)) {
            File imageFile = NNStorage.getImageFile(sd, nnf, txid);
            try {
                MD5FileUtils.saveMD5File(imageFile, digest);
            }
            catch (IOException ioe) {
                badSds.add(sd);
            }
        }
        this.storage.reportErrorsOnDirectories(badSds);
        CheckpointFaultInjector.getInstance().afterMD5Rename();
        this.renameCheckpoint(txid, NNStorage.NameNodeFile.IMAGE_NEW, nnf, false);
        if (txid > this.storage.getMostRecentCheckpointTxId()) {
            this.storage.setMostRecentCheckpointInfo(txid, Time.now());
        }
        this.initNewDirs();
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.editLog != null) {
            this.getEditLog().close();
        }
        this.storage.close();
    }

    static Collection<URI> getCheckpointDirs(Configuration conf, String defaultValue) {
        Collection dirNames = conf.getTrimmedStringCollection("dfs.namenode.checkpoint.dir");
        if (dirNames.size() == 0 && defaultValue != null) {
            dirNames.add(defaultValue);
        }
        return Util.stringCollectionAsURIs(dirNames);
    }

    static List<URI> getCheckpointEditsDirs(Configuration conf, String defaultName) {
        Collection dirNames = conf.getTrimmedStringCollection("dfs.namenode.checkpoint.edits.dir");
        if (dirNames.size() == 0 && defaultName != null) {
            dirNames.add(defaultName);
        }
        return Util.stringCollectionAsURIs(dirNames);
    }

    public NNStorage getStorage() {
        return this.storage;
    }

    public int getLayoutVersion() {
        return this.storage.getLayoutVersion();
    }

    public int getNamespaceID() {
        return this.storage.getNamespaceID();
    }

    public String getClusterID() {
        return this.storage.getClusterID();
    }

    public String getBlockPoolID() {
        return this.storage.getBlockPoolID();
    }

    public synchronized long getLastAppliedTxId() {
        return this.lastAppliedTxId;
    }

    public long getLastAppliedOrWrittenTxId() {
        return Math.max(this.lastAppliedTxId, this.editLog != null ? this.editLog.getLastWrittenTxIdWithoutLock() : 0L);
    }

    public long getCorrectLastAppliedOrWrittenTxId() {
        return Math.max(this.lastAppliedTxId, this.editLog != null ? this.editLog.getLastWrittenTxId() : 0L);
    }

    public void updateLastAppliedTxIdFromWritten() {
        this.lastAppliedTxId = this.editLog.getLastWrittenTxId();
    }

    public long getMostRecentCheckpointTxId() {
        return this.storage.getMostRecentCheckpointTxId();
    }

    private class FSImageSaver
    implements Runnable {
        private final SaveNamespaceContext context;
        private final Storage.StorageDirectory sd;
        private final NNStorage.NameNodeFile nnf;

        public FSImageSaver(SaveNamespaceContext context, Storage.StorageDirectory sd, NNStorage.NameNodeFile nnf) {
            this.context = context;
            this.sd = sd;
            this.nnf = nnf;
        }

        @Override
        public void run() {
            try {
                FSImage.this.saveFSImage(this.context, this.sd, this.nnf);
            }
            catch (SaveNamespaceCancelledException snce) {
                LOG.info((Object)("Cancelled image saving for " + this.sd.getRoot() + ": " + snce.getMessage()));
            }
            catch (Throwable t) {
                LOG.error((Object)("Unable to save image for " + this.sd.getRoot()), t);
                this.context.reportErrorOnStorageDirectory(this.sd);
            }
        }

        public String toString() {
            return "FSImageSaver for " + this.sd.getRoot() + " of type " + this.sd.getStorageDirType();
        }
    }
}

