/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.s3guard;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.s3a.S3AFileStatus;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.s3guard.DirListingMetadata;
import org.apache.hadoop.fs.s3a.s3guard.DynamoDBMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.LocalMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.MetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.NullMetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.PathMetadata;
import org.apache.hadoop.fs.shell.CommandFormat;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class S3GuardTool
extends Configured
implements Tool {
    private static final Logger LOG = LoggerFactory.getLogger(S3GuardTool.class);
    private static final String NAME = "s3guard";
    private static final String COMMON_USAGE = "When possible and not overridden by more specific options, metadata\nrepository information will be inferred from the S3A URL (if provided)\n\nGeneric options supported are:\n  -conf <config file> - specify an application configuration file\n  -D <property=value> - define a value for a given property\n";
    private static final String USAGE = "s3guard [command] [OPTIONS] [s3a://BUCKET]\n\nCommands: \n\tinit - initialize metadata repository\n\tdestroy - destroy metadata repository\n\timport - import metadata from existing S3 data\n\tdiff - report on delta between S3 and repository\n\tprune - truncate older metadata from repository\n";
    static final int SUCCESS = 0;
    static final int INVALID_ARGUMENT = 1;
    static final int ERROR = 99;
    protected S3AFileSystem s3a;
    protected MetadataStore ms;
    protected CommandFormat commandFormat = new CommandFormat(0, Integer.MAX_VALUE, new String[0]);
    private static final String META_FLAG = "meta";
    private static final String DAYS_FLAG = "days";
    private static final String HOURS_FLAG = "hours";
    private static final String MINUTES_FLAG = "minutes";
    private static final String SECONDS_FLAG = "seconds";
    private static final String REGION_FLAG = "region";
    private static final String READ_FLAG = "read";
    private static final String WRITE_FLAG = "write";
    private static S3GuardTool cmd;

    public abstract String getUsage();

    public S3GuardTool(Configuration conf) {
        super(conf);
        this.commandFormat.addOptionWithValue(META_FLAG);
        this.commandFormat.addOptionWithValue(REGION_FLAG);
    }

    abstract String getName();

    @VisibleForTesting
    public MetadataStore getMetadataStore() {
        return this.ms;
    }

    boolean parseDynamoDBRegion(List<String> paths) throws IOException {
        boolean hasS3Path;
        Configuration conf = this.getConf();
        String fromCli = this.commandFormat.getOptValue(REGION_FLAG);
        String fromConf = conf.get("fs.s3a.s3guard.ddb.region");
        boolean bl = hasS3Path = !paths.isEmpty();
        if (fromCli != null) {
            if (fromCli.isEmpty()) {
                System.err.println("No region provided with -region flag");
                return false;
            }
            if (hasS3Path) {
                System.err.println("Providing both an S3 path and the -region flag is not supported. If you need to specify a different region than the S3 bucket, configure fs.s3a.s3guard.ddb.region");
                return false;
            }
            conf.set("fs.s3a.s3guard.ddb.region", fromCli);
            return true;
        }
        if (fromConf != null) {
            if (fromConf.isEmpty()) {
                System.err.printf("No region provided with config %s, %n", "fs.s3a.s3guard.ddb.region");
                return false;
            }
            return true;
        }
        if (hasS3Path) {
            String s3Path = paths.get(0);
            this.initS3AFileSystem(s3Path);
            return true;
        }
        System.err.println("No region found from -region flag, config, or S3 bucket");
        return false;
    }

    MetadataStore initMetadataStore(boolean forceCreate) throws IOException {
        Configuration conf;
        block14: {
            block13: {
                if (this.ms != null) {
                    return this.ms;
                }
                conf = this.s3a == null ? this.getConf() : this.s3a.getConf();
                String metaURI = this.commandFormat.getOptValue(META_FLAG);
                if (metaURI == null || metaURI.isEmpty()) break block13;
                URI uri = URI.create(metaURI);
                LOG.info("create metadata store: {}", (Object)(uri + " scheme: " + uri.getScheme()));
                switch (uri.getScheme().toLowerCase()) {
                    case "local": {
                        this.ms = new LocalMetadataStore();
                        break;
                    }
                    case "dynamodb": {
                        this.ms = new DynamoDBMetadataStore();
                        conf.set("fs.s3a.s3guard.ddb.table", uri.getAuthority());
                        if (forceCreate) {
                            conf.setBoolean("fs.s3a.s3guard.ddb.table.create", true);
                            break;
                        }
                        break block14;
                    }
                    default: {
                        throw new IOException(String.format("Metadata store %s is not supported", uri));
                    }
                }
                break block14;
            }
            this.ms = new DynamoDBMetadataStore();
            if (forceCreate) {
                conf.setBoolean("fs.s3a.s3guard.ddb.table.create", true);
            }
        }
        if (this.s3a == null) {
            this.ms.initialize(conf);
        } else {
            this.ms.initialize(this.s3a);
        }
        LOG.info("Metadata store {} is initialized.", (Object)this.ms);
        return this.ms;
    }

    void initS3AFileSystem(String path) throws IOException {
        URI uri;
        try {
            uri = new URI(path);
        }
        catch (URISyntaxException e) {
            throw new IOException(e);
        }
        Configuration conf = this.getConf();
        conf.setClass("fs.s3a.metadatastore.impl", NullMetadataStore.class, MetadataStore.class);
        FileSystem fs = FileSystem.get((URI)uri, (Configuration)this.getConf());
        if (!(fs instanceof S3AFileSystem)) {
            throw new IOException(String.format("URI %s is not a S3A file system: %s", uri, fs.getClass().getName()));
        }
        this.s3a = (S3AFileSystem)fs;
    }

    List<String> parseArgs(String[] args) {
        return this.commandFormat.parse(args, 1);
    }

    private static void printHelp() {
        if (cmd == null) {
            System.err.println("Usage: hadoop s3guard [command] [OPTIONS] [s3a://BUCKET]\n\nCommands: \n\tinit - initialize metadata repository\n\tdestroy - destroy metadata repository\n\timport - import metadata from existing S3 data\n\tdiff - report on delta between S3 and repository\n\tprune - truncate older metadata from repository\n");
            System.err.println("\tperform metadata store administrative commands for s3a filesystem.");
        } else {
            System.err.println("Usage: hadoop " + cmd.getUsage());
        }
        System.err.println();
        System.err.println(COMMON_USAGE);
    }

    public static int run(String[] args, Configuration conf) throws Exception {
        String subCommand;
        String[] otherArgs = new GenericOptionsParser(conf, args).getRemainingArgs();
        if (otherArgs.length == 0) {
            S3GuardTool.printHelp();
            return 1;
        }
        switch (subCommand = otherArgs[0]) {
            case "init": {
                cmd = new Init(conf);
                break;
            }
            case "destroy": {
                cmd = new Destroy(conf);
                break;
            }
            case "import": {
                cmd = new Import(conf);
                break;
            }
            case "diff": {
                cmd = new Diff(conf);
                break;
            }
            case "prune": {
                cmd = new Prune(conf);
                break;
            }
            default: {
                S3GuardTool.printHelp();
                return 1;
            }
        }
        return ToolRunner.run((Configuration)conf, (Tool)cmd, (String[])otherArgs);
    }

    public static void main(String[] args) throws Exception {
        try {
            int ret = S3GuardTool.run(args, new Configuration());
            System.exit(ret);
        }
        catch (CommandFormat.UnknownOptionException e) {
            System.err.println(e.getMessage());
            S3GuardTool.printHelp();
            System.exit(1);
        }
        catch (Exception e) {
            e.printStackTrace(System.err);
            System.exit(99);
        }
    }

    static class Prune
    extends S3GuardTool {
        private static final String NAME = "prune";
        public static final String PURPOSE = "truncate older metadata from repository";
        private static final String USAGE = "prune [OPTIONS] [s3a://BUCKET]\n\ttruncate older metadata from repository\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        Prune(Configuration conf) {
            super(conf);
            this.commandFormat.addOptionWithValue(S3GuardTool.DAYS_FLAG);
            this.commandFormat.addOptionWithValue(S3GuardTool.HOURS_FLAG);
            this.commandFormat.addOptionWithValue(S3GuardTool.MINUTES_FLAG);
            this.commandFormat.addOptionWithValue(S3GuardTool.SECONDS_FLAG);
        }

        @VisibleForTesting
        void setMetadataStore(MetadataStore ms) {
            Preconditions.checkNotNull((Object)ms);
            this.ms = ms;
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        private long getDeltaComponent(TimeUnit unit, String arg) {
            String raw = this.commandFormat.getOptValue(arg);
            if (raw == null || raw.isEmpty()) {
                return 0L;
            }
            Long parsed = Long.parseLong(raw);
            return unit.toMillis(parsed);
        }

        @VisibleForTesting
        public int run(String[] args, PrintStream out) throws InterruptedException, IOException {
            List<String> paths = this.parseArgs(args);
            if (!this.parseDynamoDBRegion(paths)) {
                System.err.println(USAGE);
                return 1;
            }
            this.initMetadataStore(false);
            Configuration conf = this.getConf();
            long confDelta = conf.getLong("fs.s3a.s3guard.cli.prune.age", 0L);
            long cliDelta = 0L;
            cliDelta += this.getDeltaComponent(TimeUnit.DAYS, S3GuardTool.DAYS_FLAG);
            cliDelta += this.getDeltaComponent(TimeUnit.HOURS, S3GuardTool.HOURS_FLAG);
            cliDelta += this.getDeltaComponent(TimeUnit.MINUTES, S3GuardTool.MINUTES_FLAG);
            if (confDelta <= 0L && (cliDelta += this.getDeltaComponent(TimeUnit.SECONDS, S3GuardTool.SECONDS_FLAG)) <= 0L) {
                System.err.println("You must specify a positive age for metadata to prune.");
            }
            long delta = confDelta;
            if (cliDelta > 0L) {
                delta = cliDelta;
            }
            long now = System.currentTimeMillis();
            long divide = now - delta;
            this.ms.prune(divide);
            out.flush();
            return 0;
        }

        public int run(String[] args) throws InterruptedException, IOException {
            return this.run(args, System.out);
        }
    }

    static class Diff
    extends S3GuardTool {
        private static final String NAME = "diff";
        public static final String PURPOSE = "report on delta between S3 and repository";
        private static final String USAGE = "diff [OPTIONS] s3a://BUCKET\n\treport on delta between S3 and repository\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";
        private static final String SEP = "\t";
        static final String S3_PREFIX = "S3";
        static final String MS_PREFIX = "MS";

        Diff(Configuration conf) {
            super(conf);
        }

        @VisibleForTesting
        void setMetadataStore(MetadataStore ms) {
            Preconditions.checkNotNull((Object)ms);
            this.ms = ms;
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        private static String formatFileStatus(FileStatus status) {
            return String.format("%s%s%d%s%s", status.isDirectory() ? "D" : "F", SEP, status.getLen(), SEP, status.getPath().toString());
        }

        private static boolean differ(FileStatus thisOne, FileStatus thatOne) {
            Preconditions.checkArgument((thisOne != null || thatOne != null ? 1 : 0) != 0);
            return thisOne == null || thatOne == null || thisOne.getLen() != thatOne.getLen() || thisOne.isDirectory() != thatOne.isDirectory() || !thisOne.isDirectory() && thisOne.getModificationTime() != thatOne.getModificationTime();
        }

        private static void printDiff(FileStatus msStatus, FileStatus s3Status, PrintStream out) {
            Preconditions.checkArgument((msStatus != null || s3Status != null ? 1 : 0) != 0);
            if (msStatus != null && s3Status != null) {
                Preconditions.checkArgument((boolean)msStatus.getPath().equals((Object)s3Status.getPath()), (Object)String.format("The path from metadata store and s3 are different: ms=%s s3=%s", msStatus.getPath(), s3Status.getPath()));
            }
            if (Diff.differ(msStatus, s3Status)) {
                if (s3Status != null) {
                    out.printf("%s%s%s%n", S3_PREFIX, SEP, Diff.formatFileStatus(s3Status));
                }
                if (msStatus != null) {
                    out.printf("%s%s%s%n", MS_PREFIX, SEP, Diff.formatFileStatus(msStatus));
                }
            }
        }

        private void compareDir(FileStatus msDir, FileStatus s3Dir, PrintStream out) throws IOException {
            DirListingMetadata dirMeta;
            Preconditions.checkArgument((msDir != null || s3Dir != null ? 1 : 0) != 0);
            if (msDir != null && s3Dir != null) {
                Preconditions.checkArgument((boolean)msDir.getPath().equals((Object)s3Dir.getPath()), (Object)String.format("The path from metadata store and s3 are different: ms=%s s3=%s", msDir.getPath(), s3Dir.getPath()));
            }
            HashMap<Path, FileStatus> s3Children = new HashMap<Path, FileStatus>();
            if (s3Dir != null && s3Dir.isDirectory()) {
                for (FileStatus status : this.s3a.listStatus(s3Dir.getPath())) {
                    s3Children.put(status.getPath(), status);
                }
            }
            HashMap<Path, FileStatus> msChildren = new HashMap<Path, FileStatus>();
            if (msDir != null && msDir.isDirectory() && (dirMeta = this.ms.listChildren(msDir.getPath())) != null) {
                for (PathMetadata meta : dirMeta.getListing()) {
                    FileStatus status = meta.getFileStatus();
                    msChildren.put(status.getPath(), status);
                }
            }
            HashSet allPaths = new HashSet(s3Children.keySet());
            allPaths.addAll(msChildren.keySet());
            for (Path path : allPaths) {
                FileStatus s3Status = (FileStatus)s3Children.get(path);
                FileStatus msStatus = (FileStatus)msChildren.get(path);
                Diff.printDiff(msStatus, s3Status, out);
                if ((s3Status == null || !s3Status.isDirectory()) && (msStatus == null || !msStatus.isDirectory())) continue;
                this.compareDir(msStatus, s3Status, out);
            }
            out.flush();
        }

        private void compareRoot(Path path, PrintStream out) throws IOException {
            Path qualified = this.s3a.qualify(path);
            FileStatus s3Status = null;
            try {
                s3Status = this.s3a.getFileStatus(qualified);
            }
            catch (FileNotFoundException e) {
                // empty catch block
            }
            PathMetadata meta = this.ms.get(qualified);
            FileStatus msStatus = meta != null && !meta.isDeleted() ? meta.getFileStatus() : null;
            this.compareDir(msStatus, s3Status, out);
        }

        @VisibleForTesting
        public int run(String[] args, PrintStream out) throws IOException {
            URI uri;
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                out.println(USAGE);
                return 1;
            }
            String s3Path = paths.get(0);
            this.initS3AFileSystem(s3Path);
            this.initMetadataStore(true);
            try {
                uri = new URI(s3Path);
            }
            catch (URISyntaxException e) {
                throw new IOException(e);
            }
            Path root = uri.getPath().isEmpty() ? new Path("/") : new Path(uri.getPath());
            root = this.s3a.qualify(root);
            this.compareRoot(root, out);
            out.flush();
            return 0;
        }

        public int run(String[] args) throws IOException {
            return this.run(args, System.out);
        }
    }

    static class Import
    extends S3GuardTool {
        private static final String NAME = "import";
        public static final String PURPOSE = "import metadata from existing S3 data";
        private static final String USAGE = "import [OPTIONS] [s3a://BUCKET]\n\timport metadata from existing S3 data\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";
        private final Set<Path> dirCache = new HashSet<Path>();

        Import(Configuration conf) {
            super(conf);
        }

        @VisibleForTesting
        void setMetadataStore(MetadataStore ms) {
            this.ms = ms;
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        private void putParentsIfNotPresent(FileStatus f) throws IOException {
            Preconditions.checkNotNull((Object)f);
            for (Path parent = f.getPath().getParent(); parent != null; parent = parent.getParent()) {
                if (this.dirCache.contains(parent)) {
                    return;
                }
                FileStatus dir = DynamoDBMetadataStore.makeDirStatus(parent, f.getOwner());
                this.ms.put(new PathMetadata(dir));
                this.dirCache.add(parent);
            }
        }

        private long importDir(FileStatus status) throws IOException {
            Preconditions.checkArgument((boolean)status.isDirectory());
            RemoteIterator<LocatedFileStatus> it = this.s3a.listFilesAndEmptyDirectories(status.getPath(), true);
            long items = 0L;
            while (it.hasNext()) {
                FileStatus child;
                LocatedFileStatus located = (LocatedFileStatus)it.next();
                if (located.isDirectory()) {
                    child = DynamoDBMetadataStore.makeDirStatus(located.getPath(), located.getOwner());
                    this.dirCache.add(child.getPath());
                } else {
                    child = new S3AFileStatus(located.getLen(), located.getModificationTime(), located.getPath(), located.getBlockSize(), located.getOwner());
                }
                this.putParentsIfNotPresent(child);
                this.ms.put(new PathMetadata(child));
                ++items;
            }
            return items;
        }

        public int run(String[] args) throws IOException {
            URI uri;
            List<String> paths = this.parseArgs(args);
            if (paths.isEmpty()) {
                System.err.println(this.getUsage());
                return 1;
            }
            String s3Path = paths.get(0);
            this.initS3AFileSystem(s3Path);
            try {
                uri = new URI(s3Path);
            }
            catch (URISyntaxException e) {
                throw new IOException(e);
            }
            String filePath = uri.getPath();
            if (filePath.isEmpty()) {
                filePath = "/";
            }
            Path path = new Path(filePath);
            FileStatus status = this.s3a.getFileStatus(path);
            this.initMetadataStore(false);
            long items = 1L;
            if (status.isFile()) {
                PathMetadata meta = new PathMetadata(status);
                this.ms.put(meta);
            } else {
                items = this.importDir(status);
            }
            System.out.printf("Inserted %d items into Metadata Store%n", items);
            return 0;
        }
    }

    static class Destroy
    extends S3GuardTool {
        private static final String NAME = "destroy";
        public static final String PURPOSE = "destroy metadata repository";
        private static final String USAGE = "destroy [OPTIONS] [s3a://BUCKET]\n\tdestroy metadata repository\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        Destroy(Configuration conf) {
            super(conf);
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        public int run(String[] args) throws IOException {
            List<String> paths = this.parseArgs(args);
            if (!this.parseDynamoDBRegion(paths)) {
                System.err.println(USAGE);
                return 1;
            }
            this.initMetadataStore(false);
            Preconditions.checkState((this.ms != null ? 1 : 0) != 0, (Object)"Metadata store is not initialized");
            this.ms.destroy();
            LOG.info("Metadata store is deleted.");
            return 0;
        }
    }

    static class Init
    extends S3GuardTool {
        private static final String NAME = "init";
        public static final String PURPOSE = "initialize metadata repository";
        private static final String USAGE = "init [OPTIONS] [s3a://BUCKET]\n\tinitialize metadata repository\n\nCommon options:\n  -meta URL - Metadata repository details (implementation-specific)\n\nAmazon DynamoDB-specific options:\n  -region REGION - Service region for connections\n  -read UNIT - Provisioned read throughput units\n  -write UNIT - Provisioned write through put units\n\n  URLs for Amazon DynamoDB are of the form dynamodb://TABLE_NAME.\n  Specifying both the -region option and an S3A path\n  is not supported.";

        Init(Configuration conf) {
            super(conf);
            this.commandFormat.addOptionWithValue(S3GuardTool.READ_FLAG);
            this.commandFormat.addOptionWithValue(S3GuardTool.WRITE_FLAG);
        }

        @Override
        String getName() {
            return NAME;
        }

        @Override
        public String getUsage() {
            return USAGE;
        }

        public int run(String[] args) throws IOException {
            String writeCap;
            List<String> paths = this.parseArgs(args);
            String readCap = this.commandFormat.getOptValue(S3GuardTool.READ_FLAG);
            if (readCap != null && !readCap.isEmpty()) {
                int readCapacity = Integer.parseInt(readCap);
                this.getConf().setInt("fs.s3a.s3guard.ddb.table.capacity.read", readCapacity);
            }
            if ((writeCap = this.commandFormat.getOptValue(S3GuardTool.WRITE_FLAG)) != null && !writeCap.isEmpty()) {
                int writeCapacity = Integer.parseInt(writeCap);
                this.getConf().setInt("fs.s3a.s3guard.ddb.table.capacity.write", writeCapacity);
            }
            if (!this.parseDynamoDBRegion(paths)) {
                System.err.println(USAGE);
                return 1;
            }
            this.initMetadataStore(true);
            return 0;
        }
    }
}

